/*
 *========================================================================
 * $Id: benchml.c,v 1.4 2005/03/10 15:45:09 rgb Exp $
 *
 * See copyright in copyright.h and the accompanying file COPYING
 *========================================================================
 */


#include "benchmaster.h"

 /* 
  * Global pointer to the xml doc in which we build and manipulate
  * the xml image of proc and other stats until we send it.  Note
  * that it is a pointer to memory that must be freed after EACH USE
  * or we'll leak sieve-like.
  */
 xmlDocPtr doc;
 xmlNodePtr docroot;
 char outbuf[K64];
 char hostname[K];

 /* 
  * This defines some flags to control just what the daemon sends and how.
  * Currently only controls xml stuff for prettiness.
  */
 typedef struct {

   int compress;	/* Do we compress xml output? */
   int whitespace;	/* Do we squeeze out whitespace? */

 } Dctl;
 Dctl dctl;		/* daemon control flags */
 Dctl dctl_save;	/* a place to save them during a sendall */


 /*
  * This creates the basic benchxml document and fills it with host info.
  * I suspect that I'll be really, really glad if I separate this into:
  *   a) Create benchml (toplevel xml doc with toplevel tag,
  * timestamped and with benchmaster version).
  *   b) Create hostinfo tag within toplevel document.
  *   c) Create benchmark tag within toplevel document.
  *   d) Destroy benchml document so it doesn't leak memory.
  *
  * This would be far more portable and would permit multiple benchmark
  * runs to occur within a single benchml document.
  */

void create_benchml()
{

 struct timeval tv;
 /*
  * Tags that will eventually contain tags need their own name; otherwise
  * a default (tag) will do.
  */
 xmlNodePtr benchmaster,tag;


 /*
  * No compression, keep blanks for maximum readability.  This isn't a
  * loop-polled document -- one shot and done, so overhead is irrelevant.
  */
 xmlSetCompressMode(0);
 xmlKeepBlanksDefault(0);


 /* 
  * Create/allocate new xml document with version 1.0.  This document is 
  * to be freed EACH TIME after it is sent.  Should check to see if 1.0
  * is still appropriate feed here.
  */
 doc = xmlNewDoc("1.0");

 /*
  * Create toplevel/root node as <benchml>.  This identifies the
  * document type to the receiver.
  */
 docroot = xmlDocSetRootElement(doc,xmlNewDocNode(doc, NULL, "benchml", NULL));
 docroot = xmlDocGetRootElement(doc);

 /*
  * In this routine, only fill in one piece of actual content -- the
  * name/version of the benchmark itself.  Timestamp it.
  */
 sprintf(outbuf,"Benchmaster %d.%.1f",VERSION_MAJOR,VERSION_MINOR);
 benchmaster = xmlNewChild(docroot, NULL, "version", outbuf);

}

void create_hostinfo()
{

 /*
  * Tags that will eventually contain tags need their own name; otherwise
  * a default (tag) will do.
  */
 xmlNodePtr hostinfo,tag;


 /*
  * This is a toplevel tag under <benchml>, the docroot tag
  */
 hostinfo = xmlNewChild(docroot, NULL, "hostinfo", NULL);

 /*
  * These are all content tags.  Note that this is all pretty trivial,
  * and certainly no worse than a string of printf's
  */
 tag = xmlNewChild(hostinfo,NULL,"hostname",host.hostname);
 tag = xmlNewChild(hostinfo,NULL,"vendor_id",host.vendor_id);
 tag = xmlNewChild(hostinfo,NULL,"CPU name",host.name);
 tag = xmlNewChild(hostinfo,NULL,"CPU clock",host.speed);
 xmlSetProp(tag,"units","Mhz");
 /*
  * Note -- strip off the "KB" into "units" content in the original
  * parse and struct -- original struct needs to parse value and units
  * into separate variables from /proc/cpuinfo data.  Ditto for
  * memory and other system info that might be added below.  That is,
  * should be a 1:1 correspondance between elements in the host data
  * struct and output fields here, and it is probably worth changing
  * "host" to "hostinfo" throughout (renaming the struct).
  */
 tag = xmlNewChild(hostinfo,NULL,"l2cache",host.l2cache);
 xmlSetProp(tag,"units","KB");
 tag = xmlNewChild(hostinfo,NULL,"memtotal",host.memtotal);
 xmlSetProp(tag,"units","KB");
 tag = xmlNewChild(hostinfo,NULL,"memfree",host.memfree);
 xmlSetProp(tag,"units","KB");
 tag= xmlNewChild(hostinfo,NULL,"nanotimer",nanotimer.name);
 sprintf(outbuf,"%6.3f",nanotimer.nsec_granularity);
 tag = xmlNewChild(hostinfo,NULL,"nanotimer_granularity",outbuf);
 xmlSetProp(tag,"units","nsec");

}

void create_benchmark(Test *mytest)
{

 /*
  * Tags that will eventually contain tags need their own name; otherwise
  * a default (tag) will do.
  */
 xmlNodePtr benchmark,tag;

 /*
  * This is a toplevel tag under <benchml>, the docroot tag
  */
 benchmark = xmlNewChild(docroot, NULL, "benchmark", NULL);

 /*
  * These are the content tags.  Simple, simple simple.
  */
 tag = xmlNewChild(benchmark,NULL,"name",test[testnum].name);

 /*
  * This is FAKE and needs to be done RIGHT in parsecl (that is, need to
  * make a private global copy of argv before parsing, something I've done
  * before).
  */
 /* Command itself (including any relative path) */
 tag= xmlNewChild(benchmark,NULL,"command",command);
 /* Actual argv argument string */
 tag = xmlNewChild(benchmark,NULL,"args",args);
 /* Short description of benchmark (could be URL?) */ 
 sprintf(outbuf,"%s",test[testnum].about);
 tag = xmlNewChild(benchmark,NULL,"description",outbuf);
 /* Number of iterations in main (full) sample loop. */
 sprintf(outbuf,"%u",mytest->full_iter);
 tag = xmlNewChild(benchmark,NULL,"iterations",outbuf);

 

 /*
  * some parameters of interest?  We need to wrap these up so they
  * only display at all if they are relevant.
  */
 if(mytest->vector){
   /* size (vector length) */
   sprintf(outbuf,"%d",size);
   tag = xmlNewChild(benchmark,NULL,"size",outbuf);
   /* stride */
   if(random_flag && mytest->ranok){
     sprintf(outbuf,"%s","shuffled");
     tag = xmlNewChild(benchmark,NULL,"stride",outbuf);
   } else {
     sprintf(outbuf,"%d",stride);
     tag = xmlNewChild(benchmark,NULL,"stride",outbuf);
   }
 }

 /*
  * Results
  */
 sprintf(outbuf,"%8.2e",mytest->avg_time);
 tag = xmlNewChild(benchmark,NULL,"time",outbuf);
 xmlSetProp(tag,"units","nsec");
 sprintf(outbuf,"%8.2e",mytest->sigma);
 tag = xmlNewChild(benchmark,NULL,"time_stddev",outbuf);
 xmlSetProp(tag,"units","nsec");
 sprintf(outbuf,"%8.2e",mytest->min_time);
 tag = xmlNewChild(benchmark,NULL,"min_time",outbuf);
 xmlSetProp(tag,"units","nsec");
 sprintf(outbuf,"%8.2e",mytest->max_time);
 tag = xmlNewChild(benchmark,NULL,"max_time",outbuf);
 xmlSetProp(tag,"units","nsec");
 sprintf(outbuf,"%8.2e",mytest->avg_megarate);
 tag = xmlNewChild(benchmark,NULL,"rate",outbuf);
 xmlSetProp(tag,"units","10e+6");

}

void destroy_benchml()
{

 int docsize;
 xmlChar *dumpbuf;

 /*
  * We return the xml by dumping the document to a memory buffer
  * and then send the buffer, preceeded by an html-1.1 standard
  * header specifying the length, to the outgoing socket/file descriptor.
  * We have to free doc after the dump, since otherwise the daemon
  * leaks like a piddly puppy.  Not that this matters in THIS context where
  * only one document will be created/returned per invocation.
  */
 xmlDocDumpFormatMemory(doc,&dumpbuf,&docsize,1);
 if((verbose == VERBOSE) || (verbose == V_SHOW)){
   printf("V_SHOW: Sending message of length %d:\n%s",
       docsize,dumpbuf);
 }
 xmlFreeDoc(doc);

 /*
  * Send the xmlified message, already...
  */
 fprintf(OUTFP,"Content-Length: %d\n\n",docsize);
 fprintf(OUTFP,"%s",dumpbuf);

}

