/*
* $Id: set_iter.c,v 1.8 2004/06/15 11:01:46 rgb Exp $
*
* See copyright in copyright.h and the accompanying file COPYING
*
*/

#include "cpu_rate.h"

/*
 *========================================================================
 * set_iter routine automagically sets the empty and full iteration count
 * to be used in a timing test, assuming that it isn't set on the command
 * line.
 *========================================================================
 */
void set_iter(Test *mytest)
{

 int i,j,k;	/* Loop variables */
 double test_nanotime;

 if(verbose == V_SET_ITER || verbose == VERBOSE) {
   printf("#========================================================================\n");
   printf("# Setting the iteration counts for %s\n",mytest->name);
 }

 /*
  * Always begin by allocating any space required for running THIS
  * test.
  */
 if(verbose == V_SET_ITER || verbose == VERBOSE) {
   printf("# measure_rate: Entering mytest->alloc()\n");
 }
 mytest->alloc();
 

 if(iterations>0) {
   mytest->empty_iter = iterations;
   mytest->full_iter = iterations;
   if(verbose == V_SET_ITER || verbose == VERBOSE) {
     printf("# command line empty_iter = %d\n",mytest->empty_iter);
     printf("# command line full_iter = %d\n",mytest->full_iter);
     printf("#========================================================================\n");
   }
 } else {
   /*
    *========================================================================
    * Next we autotune the full timing loop so that it contains enough
    * iterations to ALSO give a "decent" precision.
    *========================================================================
    */
   test_nanotime = 0.0;
   mytest->empty_iter = 1;
   time_limit_empty = time_limit_base;
   /*
    * We want to spin as long as possible that is less than one clock tick
    * on the system, typically 0.01 seconds.  To make this really stable,
    * we need to redo each level until we KNOW what the correct
    * doubling time is.  Still, this should be good enough.
    */   
   while(test_nanotime < time_limit_empty){

     mytest->empty_iter*=2;	/* double until we get there */
     test_nanotime = time_empty(mytest);
     if(verbose == V_SET_ITER || verbose == VERBOSE) {
       printf("# set_iter(): mytest->empty_iter = %d, empty loop test time  = %e (nsec)\n",mytest->empty_iter,test_nanotime);
     }

   }
   /*
    * remember the actual value here, and use IT to determine out of
    * bounds results.
    */
   time_limit_empty = test_nanotime;
   if(verbose == V_SET_ITER || verbose == VERBOSE) {
     printf("Estimate of empty time is %f\n",test_nanotime/(double)mytest->empty_iter);
   }

   /*
    *========================================================================
    * Next we autotune the full timing loop so that it contains enough
    * iterations to ALSO give a "decent" precision.
    *========================================================================
    */
   test_nanotime = 0.0;
   mytest->full_iter = 1;	/* I want at LEAST 2 iterations */
   time_limit_full = time_limit_base;
   /*
    * We want to spin as long as possible that is less than one clock tick
    * on the system, typically 0.01 seconds.  However, now we have to
    * be careful, as if the basic task takes longer than time_limit_full
    * already, it might also take longer than 0.01 seconds.  In this case
    * we need to relax the idea that clock interrupts inside the loop
    * should not be counted and just average over them.  Sigh.
    */
   while(test_nanotime < time_limit_full){
     mytest->full_iter*=2;	/* double until we get there */
     test_nanotime = time_full(mytest);
     if(verbose == V_SET_ITER || verbose == VERBOSE) {
       printf(" set_iter(): mytest->full_iter = %d, full loop test time = %e (nsec)\n",mytest->full_iter,test_nanotime);
     }
   }
   /*
    * remember the actual value here, and use IT to determine out of
    * bounds results.  Note that it should be a highball estimate.
    */
   time_limit_full = test_nanotime;
   if(verbose == V_SET_ITER || verbose == VERBOSE) {
     printf("Estimate of full time is %f\n",nanotimer.delta/(double)mytest->full_iter);
   }
   
 }

 /*
  * Always end by freeing any space allocated above for THIS test.
  */
 mytest->free();
 
}
